;==============================================================================;
;==	Bootloader										Create Date 2018/02/12
;==	pic16f145x_family								Last Update 2018/04/03
;==	file name	: bootloader.asm							Saka Softwares
;==============================================================================;
;== 割り込みは一切使いません。
;== マイクロチップ社製ブートローダと互換があります。(多少違いはありますが…)
;== Ver1.02に対応しました。0x500Word以内に収まりました。
;== Include ===================================================================;
	#define _HID_BOOTLOADER
	#include "usbpic16.inc"
	#include "usbhiddrv.inc"
	#include "bootloader.inc"

	radix	dec

;== Constants =================================================================;
#define TRUE	1
#define FALSE	0
; The bootloader version, which the bootloader PC application can do extended query to get.
; Value provided is expected to be in the format of BOOTLOADER_VERSION_MAJOR.BOOTLOADER_VERSION_MINOR
; Ex: 1.01 would be BOOTLOADER_VERSION_MAJOR == 1, and BOOTLOADER_VERSION_MINOR == 1
#define BOOTLOADER_VERSION_MAJOR	1 ; Legal value 0-255
#define BOOTLOADER_VERSION_MINOR	2 ; Legal value 0-99.  (1 = X.01)

; Section defining the address range to erase for the erase device command, along with the valid programming range to be reported by the QUERY_DEVICE command.
#define PROG_MEM_START_ADDRESS		APP_SIGNATURE_ADDRESS	; Beginning of application program memory (not occupied by bootloader).  **THIS VALUE MUST BE ALIGNED WITH 64 BYTE BLOCK BOUNDRY** Also, in order to work correctly, make sure the StartPageToErase is set to erase this section.

#define QUERY_DEVICE				0x02	;Command that the host uses to learn about the device (what regions can be programmed, and what type of memory is the region)
#define UNLOCK_CONFIG				0x03	;Note, this command is used for both locking and unlocking the config bits (see the "//Unlock Configs Command Definitions" below)
#define ERASE_DEVICE				0x04	;Host sends this command to start an erase operation.  Firmware controls which pages should be erased.
#define PROGRAM_DEVICE				0x05	;If host is going to send a full RequestDataBlockSize to be programmed, it uses this command.
#define PROGRAM_COMPLETE			0x06	;If host send less than a RequestDataBlockSize to be programmed, or if it wished to program whatever was left in the buffer, it uses this command.
#define GET_DATA					0x07	;The host sends this command in order to read out memory from the device.  Used during verify (and read/export hex operations)
#define RESET_DEVICE				0x08	;Resets the microcontroller, so it can update the config bits (if they were programmed, and so as to leave the bootloader (and potentially go back into the main application)
#define SIGN_FLASH					0x09	;The host PC application should send this command after the verify operation has completed successfully.  If checksums are used instead of a true verify (due to ALLOW_GET_DATA_COMMAND being commented), then the host PC application should send SIGN_FLASH command after is has verified the checksums are as exected. The firmware will then program the SIGNATURE_WORD into flash at the SIGNATURE_ADDRESS.
#define QUERY_EXTENDED_INFO			0x0C	;Used by host PC app to get additional info about the device, beyond the basic NVM layout provided by the query device command

; Unlock Configs Command Definitions
#define UNLOCKCONFIG				0x00	;Sub-command for the ERASE_DEVICE command
#define LOCKCONFIG					0x01	;Sub-command for the ERASE_DEVICE command

; Query Device Response "Types" 
#define MEMORY_REGION_PROGRAM		0x01	;When the host sends a QUERY_DEVICE command, need to respond by populating a list of valid memory regions that exist in the device (and should be programmed)
#define MEMORY_REGION_EEDATA		0x02
#define MEMORY_REGION_CONFIG		0x03
#define MEMORY_REGION_USERID		0x04
#define MEMORY_REGION_END			0xFF	;Sort of serves as a "null terminator" like number, which denotes the end of the memory region list has been reached.
#define BOOTLOADER_NEWER_FLAG		0xA5	;Tacked on in the VersionFlag byte, to indicate when using newer version of bootloader with extended query info available

; BootState Variable States
#define IDLE						0x00
#define NOT_IDLE					0x01

; OtherConstants
#define INVALID_ADDRESS				0xFFFFFFFF
#define CORRECT_UNLOCK_KEY			0xB5	; 未使用

; Application and Microcontroller constants
#define BYTES_PER_ADDRESS_PIC16		0x01	;One byte per address.	PIC24 uses 2 bytes for each address in the hex file.
#define USB_PACKET_SIZE				0x40
#define WORDSIZE					0x02	;PIC18 uses 2 byte words, PIC24 uses 3 byte words.
#define RequestDataBlockSize		0x3A	;Number of data bytes in a standard request to the PC.	Must be an even number from 2-58 (0x02-0x3A).  Larger numbers make better use of USB bandwidth and 
											;yeild shorter program/verify times, but require more micrcontroller RAM for buffer space.

; General command (with data in it) packet structure used by PROGRAM_DEVICE and GET_DATA commands		
#define Command						0	; char
#define Address						1	; long
#define Adr8						1	; cahr (Address)	& 0xFF
#define Adr16						2	; char (Address>>8) & 0xFF
#define Adr24						3	; char (Address>>16)& 0xFF
#define Adr32						4	; char (Address>>24)& 0xFF
#define Size						5	; char
#define Datas						6	; char [REQUEST_DATA_BLOCK_SIZE]
; This struct used for responding to QUERY_DEVICE command (on a device with four programmable sections)
;#define	Command					0	; char
#define PacketDataFieldSize			1	; char
#define BytesPerAddress				2	; char
#define Type1						3	; char
#define Address1					4	; long
#define Length1						8	; long
#define Type2						12	; char
#define Address2					13	; long
#define Length2						17	; long
#define Type3						21	; char
#define Address3					22	; long
#define Length3						26	; long
#define Type4						30	; char
#define Address4					31	; long
#define Length4						35	; long
#define Type5						39	; char
#define Address5					40	; long
#define Length5						44	; long
#define Type6						48	; char
#define Address6					49	; long
#define Length6						53	; long
#define VersionFlag					57	; char
#define ExtraPadBytes				58	; dhar [6]
; For UNLOCK_CONFIG command
;#define	Command					0	; char
#define LockValue					1	; char
; Structure for the QUERY_EXTENDED_INFO command (and response)
;#define	Command					0	; char
#define BootloaderVersion			1	; int
#define ApplicationVersion			3	; int
#define AppVerLow					3	; char (ApplicationVersion)    & 0xFF
#define AppVerHigh					4	; char (ApplicationVersion>>8) & 0xFF
#define SignatureAddress			5	; long
#define SignatureValue				9	; int
#define ErasePageSize				11	; long
#define Config1LMask				15	; char
#define Config1HMask				16	; char
#define Config2LMask				17	; char
#define Config2HMask				18	; char
#define Config3LMask				19	; char
#define Config3HMask				20	; char
#define Config4LMask				21	; char
#define Config4HMask				22	; char
#define Config5LMask				23	; char
#define Config5HMask				24	; char
#define Config6LMask				25	; char
#define Config6HMask				26	; char
#define Config7LMask				27	; char
#define Config7HMask				28	; char

;== Variables =================================================================;
BOOTVAL		UDATA	0x60				; Bank 0
;== 各変数
mBTSTSTE	res 1						; BootState
mCFGLKVAL	res 1						; ConfigsLockValue
mCOUNT		res 1						; Counter
;== 処理バッファ (Bank3 PMCOM/PMDAT/PMADRと共通)
BOOTBUF1	UDATA		0x1A0
#define		PGBUFADR	0x20F0			; Linear Address
mPGBUF		res ERASE_PAGE_SIZE * 2		; Programing Buffer (Bank3)
mPGDAT		res 2						; 読み出しデータ
mPGADR		res 2						; 書き込みアドレス
mPGPNT		res 2						; 書き込みアドレス（保存用）
mPGCOMP		res 1						; 書き込み完了フラグ
mPGCNT		res 1						; 読み出し/書き込みカウント
;== USB Buffer
BOOTBUF2	UDATA		0x220
#define		PKTToPCADR	0x2140			; Linear Address
mPKTToPC	res USB_PACKET_SIZE			; PacketToPC (Bank4)
BOOTBUF3	UDATA		0x2A0
#define		PKTFrPCADR	0x2190			; Linear Address
mPKTFrPC	res USB_PACKET_SIZE			; PacketFromPC (Bamk5)

;==============================================================================;
#define	dword(a)	de low(a), high(a)
;==============================================================================;
DEVICEDATA	CODE
;==============================================================================;
;== PICのデバイス情報 
#define PROG_MEM_LENGTH  (PROG_MEM_STOP_ADDRESS - PROG_MEM_START_ADDRESS)
DeviceDatas:
	de	QUERY_DEVICE								; Command
	de	RequestDataBlockSize						; PacketDataFieldSize
	de	BYTES_PER_ADDRESS_PIC16						; BytesPerAddress

	de	MEMORY_REGION_PROGRAM						; Type1
	dword((PROG_MEM_START_ADDRESS*2)&0xFFFF)
	dword((PROG_MEM_START_ADDRESS*2>>16)&0xFFFF)	; Address1
	dword((PROG_MEM_LENGTH*2)&0xFFFF)
	dword((PROG_MEM_LENGTH*2>>16)&0xFFFF)			; Length1

	de	MEMORY_REGION_CONFIG						; Type2
	dword((CONFIG_START_ADDRESS*2)&0xFFFF)
	dword((CONFIG_START_ADDRESS*2>>16)&0xFFFF)		; Address2
	dword((CONFIG_SECTION_LENGTH*2)&0xFFFF)
	dword((CONFIG_SECTION_LENGTH*2>>16)&0xFFFF)		; Length2

	de	MEMORY_REGION_USERID						; Type3
	dword((USER_ID_ADDRESS*2)&0xFFFF)
	dword((USER_ID_ADDRESS*2>>16)&0xFFFF)			; Address3
	dword((USER_ID_SIZE*2)&0xFFFF)
	dword((USER_ID_SIZE*2>>16)&0xFFFF)				; Length3

	de	MEMORY_REGION_END							; Type4
DeviceDatas_END:
;== 拡張情報 (Bootloder Ver1.02以降)
ExtendedInfo:
	de	QUERY_EXTENDED_INFO							; Command
	de	BOOTLOADER_VERSION_MINOR
	de	BOOTLOADER_VERSION_MAJOR					; BootloaderVersion
	dword(0x0000)									; ApplicationVersion (後で)
	dword((APP_SIGNATURE_ADDRESS*2)&0xFFFF)
	dword((APP_SIGNATURE_ADDRESS*2>>16)&0xFFFF)		; SignatureAddress
	de	APP_SIGNATURE_VALUE							; SignatureValue
	de	RETLW_OPCODE_MSB							; return to OPcode
	dword((ERASE_PAGE_SIZE*2)&0xFFFF)
	dword((ERASE_PAGE_SIZE*2>>16)&0xFFFF)			; ErasePageSize
	de	0xFF										; Config1LMask
	de	0xFF										; Config1HMask
	de	0xFF										; Config2LMask
	de	0xFF										; Config2HMask
	de	0xFF										; Config3LMask
	de	0xFF										; Config3HMask
	de	0xFF										; Config4LMask
	de	0xFF										; Config4HMask
	de	0xFF										; Config5LMask
	de	0xFF										; Config5HMask
	de	0xFF										; Config6LMask
	de	0xFF										; Config6HMask
	de	0xFF										; Config7LMask
	de	0xFF										; Config7HMask
ExtendedInfo_END:

;== Declarations ==============================================================;
BOOTCODE	CODE
;==============================================================================;
;==		プートーローダー初期化部
;==============================================================================;
BootlodarInit:
		global	BootlodarInit
		movlw	TRUE
		movwf	mCFGLKVAL				; TRUE
		BANKSEL	mPGCOMP
		movwf	mPGCOMP					; TRUE
		BANKSEL	mBTSTSTE 
		clrf	mBTSTSTE				; IDLE
		return
;== BootlodarInit end
;==============================================================================;
;==		ブートローダー実行部
;==============================================================================;
BootlodarTasks:
		global	BootlodarTasks
		movf	mBTSTSTE, F
		bnz		Boot_Working			; ブートローダー処理中
;==============================================================================;
;==		受信待ち処理
;==============================================================================;
Boot_Idle:
		call	HIDRxIsBusy				; 受信したか?
		bnz		Boot_Idle_Exit			; -- return (2)
		;== 受信処理
		movlw	low  (PKTFrPCADR)
		movwf	mHIDRxADRL
		movlw	high (PKTFrPCADR)
		movwf	mHIDRxADRH
		movlw	USB_PACKET_SIZE
		movwf	mHIDRxCNT
		movlw	fPIBUSY
		movwf	mHIDRxFLAG
		call	HIDRxReport				; データ取得
		;== 送信準備
		incf	mBTSTSTE, F				; NOT_IDLE
		movlw	USB_PACKET_SIZE
		movwf	mCOUNT					; カウント
		movlw	low  (PKTToPCADR)
		movwf	FSR0L
		movlw	high (PKTToPCADR)
		movwf	FSR0H					; 送信用アドレス
BOOT_TxBUF_Clear:
		movlw	0
		movwi	FSR0++					; 送信データクリア
		decfsz	mCOUNT, F
		bra		BOOT_TxBUF_Clear
;== 戻り位置(1)
Boot_NotIdle_Exit:
		return
;== BootlodarTasks end(1)
;==============================================================================;
;==		受信処理
;==============================================================================;
Boot_Working:
		BANKSEL	mPKTFrPC
		movf	mPKTFrPC, W				; 0x00
		BANKSEL	mBTSTSTE				; -- Unknown --
		decf	WREG, F					; 0x01
										; -- Unknown --
		decf	WREG, F					; 0x02
		bz		Query_Device			; QUERY_DEVICE
		decf	WREG, F					; 0x03
		bz		Unlock_Config			; UNLOCK_CONFIG
		decf	WREG, F					; 0x04
		bz		Erase_Device			; ERASE_DEVICE
		decf	WREG, F					; 0x05
		bz		Program_Device			; PROGRAM_DEVICE
		decf	WREG, F					; 0x06
		bz		Program_Complete		; PROGRAM_COMPLETE
		decf	WREG, F					; 0x07
		bz		Get_Data				; GET_DATA
		decf	WREG, F					; 0x08
		bz		Reset_Device			; RESET_DEVICE
		decf	WREG, F					; 0x09
		bz		Sign_Flash				; SIGN_FLASH
		decf	WREG, F					; 0x0A
										; -- Unknown --
		decf	WREG, F					; 0x0B
										; -- Unknown --
		decf	WREG, F					; 0x0C
		bz		Query_Extended_Info		; QUERY_EXTENDED_INFO
;== 戻り位置(2)
Boot_Idle_Exit:
		BANKSEL	mBTSTSTE
		clrf	mBTSTSTE				; IDLE
		return
;== BootlodarTasks end(2)
;==============================================================================;
;== サイン記入処理 (Ver1.02以降)
;==============================================================================;
Sign_Flash:
		BANKSEL	PMCON1
		bcf		PMCON1, CFGS			; Flash ROMの読み出し
		;== 転送元
		movlw	low  (PROG_MEM_START_ADDRESS)
		movwf	PMADRL
		movlw	high (PROG_MEM_START_ADDRESS)
		movwf	PMADRH					; 転送元
		;== 転送先
		movlw	low  (PGBUFADR)
		movwf	FSR1L
		movlw	high (PGBUFADR)
		movwf	FSR1H					; 転送先
		movlw	ERASE_PAGE_SIZE
		movwf	mPGCNT					; 転送数
Sign_Flash_Read_Loop:
		bsf		PMCON1, RD				; 読み取り
		nop
		nop
		movf	PMDATL, W
		movwi	FSR1++					; データLow
		movf	PMDATH, W
		movwi	FSR1++					; データHifh
		incf	PMADRL, F
		btfsc	STATUS, Z
		incf	PMADRH, F				; Address +1
		decfsz	mPGCNT, F				; 転送数
		bra		Sign_Flash_Read_Loop	; データ転送
Sign_Flash_Set_Code:
		movlw	APP_SIGNATURE_VALUE
		movwf	(mPGBUF+APP_SIGN_ADRL)
		movlw	RETLW_OPCODE_MSB
		movwf	(mPGBUF+APP_SIGN_ADRH)	; サイン記入
Sign_Flash_BlockErase:
		;== イレーズ
		movlw	low  (PROG_MEM_START_ADDRESS)
		movwf	PMADRL
		movlw	high (PROG_MEM_START_ADDRESS)
		movwf	PMADRH					; イレーズアドレス
		call	ProgramFlashErase		; イレーズ
Sign_Flash_Data:
		;== 転送元
		movlw	low  (PGBUFADR)
		movwf	FSR0L
		movlw	high (PGBUFADR)
		movwf	FSR0H					; 転送元
		movlw	ERASE_PAGE_SIZE
		movwf	mPGCNT					; 転送数
Sign_Flash_Transfer:
		moviw	FSR0++
		movwf	PMDATL					; データLow
		moviw	FSR0++
		movwf	PMDATH					; データHifh
		call	ProgramFlashDatas		; データ転送
		incf	PMADRL, F				; 下位
		decfsz	mPGCNT, F				; 転送数
		bra		Sign_Flash_Transfer
Sign_Flash_Write:
		decf	PMADRL, F				; 書き込み位置修正
		call	ProgramFlashWrite		; プログラム領域に書き込み
		goto	Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;== 拡張情報の問い合わせ処理 (Ver1.02以降)
;==============================================================================;
Query_Extended_Info:
		call	HIDTxIsBusy				; 送信可能か?
		bnz		Boot_NotIdle_Exit		; -- return (1)
		;== ROMデータ転送
		movlw	low  (ExtendedInfo)
		movwf	(mDTSADR+0)
		movlw	high (ExtendedInfo)		; DeviceDatas
		movwf	(mDTSADR+1)				; 転送元
		;== 転送先
		movlw	low  (PKTToPCADR)
		movwf	(mDTDADR+0)
		movlw	high (PKTToPCADR)		; mPKTToPC (Linear Address)
		movwf	(mDTDADR+1)				; 転送先
		movlw	(ExtendedInfo_END - ExtendedInfo)
		movwf	mDTCNT					; 転送数
		call	DataTransfer			; データ転送
		;== Newバージョン設定
		BANKSEL	mPKTToPC
		movlw	low  (APP_VERSION_ADDRESS)
		movwf	FSR0L
		movlw	high (APP_VERSION_ADDRESS+0x80)	;ROMアドレスに変更
		movwf	FSR0H
		moviw	FSR0++
		movwf	(mPKTToPC + AppVerLow)	; アプリバージョン(MINOR)を記入
		moviw	FSR0++
		movwf	(mPKTToPC + AppVerHigh) ; アプリバージョン(MAJOR)を記入
		BANKSEL	mBTSTSTE
		;== USBデータ転送設定
		call	Send_PaketData			; パケット送信
		goto	Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;== デバイス情報の問い合わせ処理
;==============================================================================;
Query_Device:
		call	HIDTxIsBusy				; 送信可能か?
		bnz		Boot_NotIdle_Exit		; -- return (1)
		;== ROMデータ転送
		movlw	low  (DeviceDatas)
		movwf	(mDTSADR+0)
		movlw	high (DeviceDatas)		; DeviceDatas (ROM Address)
		movwf	(mDTSADR+1)				; 転送元
		;== 転送先
		movlw	low  (PKTToPCADR)
		movwf	(mDTDADR+0)
		movlw	high (PKTToPCADR)		; mPKTToPC (Linear Address)
		movwf	(mDTDADR+1)				; 転送先
		movlw	(DeviceDatas_END - DeviceDatas)
		movwf	mDTCNT					; 転送数
		call	DataTransfer			; データ転送
		;== Newバージョン設定
		BANKSEL	mPKTToPC
		movlw	BOOTLOADER_NEWER_FLAG	;
		movwf	(mPKTToPC + VersionFlag); Ver1.02以降に対応
		BANKSEL	mDTCNT
		;== USBデータ転送設定
		call	Send_PaketData			; パケット送信
		goto	Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		ロック/アンロック処理
;==============================================================================;
Unlock_Config:
		BANKSEL mPKTFrPC
		subwf	(mPKTFrPC + LockValue), W
		BANKSEL mCFGLKVAL
		movwf	mCFGLKVAL				; TRUE:0x01 / FALSE:0x00
		goto	Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		イレーズ処理
;==============================================================================;
Erase_Device:
		BANKSEL	mPGADR
		movlw	low  (PROG_MEM_START_ADDRESS)
		movwf	(mPGADR+0)
		movlw	high (PROG_MEM_START_ADDRESS)
		movwf	(mPGADR+1)
Prog_Erase_Loop:
		movf	(mPGADR+0), W
		movwf	PMADRL
		movf	(mPGADR+1), W
		movwf	PMADRH
		call	ProgramFlashErase		; プログラム領域の消去
		movlw	ERASE_PAGE_SIZE
		addwf	(mPGADR+0), F
		btfsc	STATUS, C
		incf	(mPGADR+1), F			; 次の消去ブロックアドレス
		movlw	high (PROG_MEM_STOP_ADDRESS)
		subwf	(mPGADR+1), W
		btfss	STATUS, C
		bra		Prog_Erase_Loop			; PROG_MEM_STOP_ADDRESS > PMADRH
UserID_Erase:
		clrf	PMADRL
		clrf	PMADRH
		call	UserIdErase				; User ID 消去
		goto	Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		ブログラム処理
;==============================================================================;
Program_Device:
		;== 転送元 ============================================================;
		BANKSEL	mPKTFrPC
		movf	(mPKTFrPC + Size), W
		BANKSEL	mPGCNT
		movwf	mPGCNT					; 転送サイズ*2
		movlw	low  (PKTFrPCADR + Datas)
		movwf	FSR0L					; 転送データアドレスLow
		movlw	high (PKTFrPCADR + Datas)
		movwf	FSR0H					; 転送データアドレスHigh
		movf	mPGCNT, W				; 転送サイズ*2
		sublw	(USB_PACKET_SIZE - Datas)
		addwf	FSR0L, F				; 後詰め処理 (上位処理は不要)
		lsrf	mPGCNT, F				; 1/2して転送数に変更
		;== 転送先 Low ========================================================;
		BANKSEL	mPKTFrPC
		movf	(mPKTFrPC + Adr8), W
		BANKSEL	mPGADR
		movwf	(mPGADR+0)				; 転送先アドレスLow*2
		;== 転送先 High =======================================================;
		BANKSEL	mPKTFrPC
		movf	(mPKTFrPC + Adr16), W
		BANKSEL	mPGADR
		movwf	(mPGADR+1)				; 転送先アドレスHigh*2
		;== 転送先 Upper ======================================================;
		BANKSEL	mPKTFrPC
		movf	(mPKTFrPC + Adr24), W
		BANKSEL	mPGADR
		rrf		WREG, F					; 転送先アドレス*2を
		rrf		(mPGADR+1), F			; 1/2にして
		rrf		(mPGADR+0), F			; 転送先アドレス を生成		
		;== 書き込み先の確認 ===================================================;
		movf	(mPGADR+0), W
		movwf	PMADRL					; 下位を設定
		movf	(mPGADR+1), W
;		movwf	PMADRH					; 上位は後で
		sublw	(high (USER_ID_ADDRESS) - 1)
		btfsc	STATUS, C
		bra		Write_FlashProg			; 0x7FFF以下なら
		movlw	low (CONFIG_START_ADDRESS)
		subwf	(mPGADR+0), W
		bnz		Write_ConfigBits		; User ID
		BANKSEL	mCFGLKVAL				; Configuration Word
		movf	mCFGLKVAL, F
		btfss	STATUS, Z				; Unlockの時
		goto	Boot_Idle_Exit			; -- return (2) Error
;== Configuration Mempryの書き込み =============================================;
Write_ConfigBits:
		BANKSEL	PMADR
		clrf	PMADRH					; 上位アドレスをクリアする
		bsf		PMCON1, CFGS			; コンフィグレーションの読み出し
Write_ConfigBits_Loop:
		BANKSEL	PMDAT
		moviw	FSR0++
		movwf	PMDATL					; 下位データ
		moviw	FSR0++
		movwf	PMDATH					; 上位データ
		call	ConfigBitsWrite			; コンフィグビットの書き込み
		incf	PMADRL, F
		decfsz	mPGCNT, F
		bra		Write_ConfigBits_Loop
		goto	Boot_Idle_Exit			; -- return (2)
;== Flash Program Mempryの書き込み =============================================;
Write_FlashProg:
		movf	mPGCOMP, F
		bnz		Write_FlashProg_Start	; 書き込み開始 (mPRGCOMP == TRUE)
		movf	(mPGPNT+0), W
		subwf	(mPGADR+0), W
		bnz		Boot_Idle_Exit			; -- return (2) Error
		movf	(mPGPNT+1), W
		subwf	(mPGADR+1), W
		bnz		Boot_Idle_Exit			; -- return (2) Error
		bra		Write_FlashProg_Restart	; 追加書き込み
Write_FlashProg_Start:
		clrf	mPGCOMP					; 書込開始(FALSE)
Write_FlashProg_Loop:
		moviw	FSR0++
		movwf	PMDATL					; データLow
		moviw	FSR0++
		movwf	PMDATH					; データHifh
		call	ProgramFlashDatas		; データ転送
		incf	(mPGADR+0), F
		btfsc	STATUS, Z
		incf	(mPGADR+1), F			; 書き込みアドレス更新
		movf	(mPGADR+0), W
		movwf	PMADRL					; 下位のみ更新
		decf	mPGCNT, F				; 転送数
Write_FlashProg_Restart:
		movf	mPGCNT, W
		bz		Write_FlashProg_Exit	; 終了
		movf	(mPGADR+0), W
		andlw	(WRITE_BLOCK_SIZE - 1)	; ブロックの境界か
		btfss	STATUS, Z
		bra		Write_FlashProg_Loop	; 繰り返し
Write_FlashProg_WR:
		decf	(mPGADR+0), W
		movwf	PMADRL					; 下位
		movf	(mPGADR+1), W
		movf	(mPGADR+0), F
		btfsc	STATUS, Z
		decf	(mPGADR+1), W
		movwf	PMADRH					; 上位
		call	ProgramFlashWrite		; プログラム領域に書き込み
		movf	(mPGADR+0), W
		movwf	PMADRL					; 下位のみ更新
		movf	mPGCNT, F
		bnz		Write_FlashProg_Loop	; データが残っている時
Write_FlashProg_Exit:
		movf	(mPGADR+0), W
		movwf	(mPGPNT+0)				; 書き込みアドレスLow  (保存)
		movf	(mPGADR+1), W
		movwf	(mPGPNT+1)				; 書き込みアドレスHigh (保存)
		goto	Boot_Idle_Exit			; -- return (2)

;==============================================================================;
;==		プログラム完了処理
;==============================================================================;
Program_Complete:
		BANKSEL	mPGCOMP
		movlw	TRUE
		movwf	mPGCOMP					; 書込終了(TRUE)
		clrf	mPGCNT					; カウント0
		movf	(mPGPNT+0), W
		movwf	(mPGADR+0)				; 書き込みアドレスLow
		movf	(mPGPNT+1), W
		movwf	(mPGADR+1)				; 書き込みアドレスHigh
		goto	Write_FlashProg_WR

;==============================================================================;
;==		データ読出し処理
;==============================================================================;
Get_Data:
		call	HIDTxIsBusy				; 送信可能か?
		bnz		Boot_NotIdle_Exit		; -- return (1)
		movlw	GET_DATA
		BANKSEL	mPKTToPC
		movwf	(mPKTToPC + Command)	; GET_DATA コマンド
		;== 転送先 Low ========================================================;
		BANKSEL	mPKTFrPC
		movf	(mPKTFrPC + Adr8), W
		BANKSEL	mPKTToPC
		movwf	(mPKTToPC + Adr8)
		BANKSEL	mPGADR
		movwf	(mPGADR+0)				; 転送先アドレスLow*2
		;== 転送先 High =======================================================;
		BANKSEL	mPKTFrPC
		movf	(mPKTFrPC + Adr16), W
		BANKSEL	mPKTToPC
		movwf	(mPKTToPC + Adr16)
		BANKSEL	mPGADR
		movwf	(mPGADR+1)				; 転送先アドレスHigh*2
		;== 転送先 Upper ======================================================;
		BANKSEL	mPKTFrPC
		movf	(mPKTFrPC + Adr24), W
		BANKSEL	mPKTToPC
		movwf	(mPKTToPC + Adr24)
		BANKSEL	mPGADR
		rrf		WREG, F					; 転送先アドレス*2を
		rrf		(mPGADR+1), F			; 1/2にして
		rrf		(mPGADR+0), F			; 転送先アドレス を生成		
		;== 転送数 ============================================================;
		BANKSEL	mPKTFrPC
		movf	(mPKTFrPC + Size), W
		BANKSEL	mPKTToPC
		movwf	(mPKTToPC + Size)
		BANKSEL	mCOUNT
		movwf	mCOUNT					; 転送サイズ*2
		;== 転送先 ============================================================;
		movlw	low  (PKTToPCADR + Datas)
		movwf	FSR1L
		movlw	High (PKTToPCADR + Datas)
		movwf	FSR1H
		movf	mCOUNT, W
		sublw	(USB_PACKET_SIZE - Datas)
		addwf	FSR1L, F				; 後詰め処理 (上位処理は不要)
		lsrf	mCOUNT, F				; 転送サイズに変更
		;== 読み出し先の確認 ===================================================;
		BANKSEL	PMADR
		movf	(mPGADR+0), W
		movwf	PMADRL
		movf	(mPGADR+1), W
		movwf	PMADRH
		bcf		PMCON1, CFGS			; Flash ROMの読み出し
		sublw	high (USER_ID_ADDRESS) - 1
		btfsc	STATUS, C
		bra		Read_FlashProg			; 0x7FFF以下なら
		clrf	PMADRH
		bsf		PMCON1, CFGS			; コンフィグレーションの読み出し
		;== Flash Mempryの読み出し =============================================;
Read_FlashProg:
		BANKSEL	PMDAT
		bsf		PMCON1, RD				; 読み取り
		nop
		nop
		movf	PMDATL, W
		movwf	(mPGDAT+0)
Read_FlashProg_Low:
		movwi	FSR1++					; データLow
		movf	PMDATH, W
		movwf	(mPGDAT+1)
		incf	(mPGDAT+0), W
		bnz		Read_FlashProg_High		; 0xFF以外
		movlw	high (BLANK_FLASH_WORD_VALUE)
		subwf	(mPGDAT+1), W
		bnz		Read_FlashProg_High		; 0x3F以外
		movlw	0xFF					; 3FFF の時は FFFF を返す
		movwf	(mPGDAT+1)
Read_FlashProg_High:
		movf	(mPGDAT+1), W
		movwi	FSR1++					; データHifh
		incf	PMADRL, F
		btfsc	STATUS, Z
		incf	PMADRH, F				; Address +1
		BANKSEL	mCOUNT
		decfsz	mCOUNT, F				; 転送数
		bra		Read_FlashProg
		call	Send_PaketData			; パケット送信
		goto	Boot_Idle_Exit			; -- return (2)
		
;==============================================================================;
;==		リセット処理	(ユーザープログラムへ移行)
;==============================================================================;
Reset_Device:
		goto	Main_Reset				; メインへ

;==============================================================================;
;==		内部処理
;==============================================================================;
;==		パケットデータの送信
;==============================================================================;
Send_PaketData:
		BANKSEL	mHIDTxADRL
		movlw	low  (PKTToPCADR)
		movwf	mHIDTxADRL
		movlw	high (PKTToPCADR)
		movwf	mHIDTxADRH
		movlw	USB_PACKET_SIZE			; 64Byte
		movwf	mHIDTxCNT
		movlw	fPIRAM | fPIBUSY
		movwf	mHIDTxFLAG
		call	HIDTxReport				; データ送信
		return

;==============================================================================;
;==		各メモリ書き換え実行部
;==============================================================================;
;== UserIDの消去
UserIdErase:
		movlw	b'01011000'				; Configuration Word / Erease Opration
		bra		Memery_Write
;== フラッシュメモリの消去
ProgramFlashErase:
		movlw	b'00011000'				; Flash Memory / Erease Opration
		bra		Memery_Write
;== UserID/Configの書き込み
ConfigBitsWrite:
		movlw	b'01001000'				; Configuration Word / Write the page
		bra		Memery_Write
;== フラッシュメモリの書き込み
ProgramFlashWrite:
		movlw	b'00001000'				; Flash Memory / Write the page
		bra		Memery_Write
;== フラッシュメモリへ転送
ProgramFlashDatas:
		movlw	b'00101000'				; Flash Memory / Latches Only
;== 書き込み処理
Memery_Write:
		movwf	PMCON1					; 設定
		clrwdt							; ウォッチドックタイマのクリア
		bsf		PMCON1, WREN			; Enable Writes
		movlw	0x55
		movwf	PMCON2
		movlw	0xAA	
		movwf	PMCON2
		bsf		PMCON1, WR				; 書き換え
		nop
		nop
		bcf		PMCON1, WREN			; Disable Write
		return
		
;==============================================================================;
		END
